//==============================================================================
// Project:		Wealth Transfers and their Economic Effects
// File name:	Australian residents - inheritance outflows - expected deaths
// Objective: 	Estimating inheritance outflows based on expected deaths for
//				people whose entire family resides in Australia.
//
// Created: 	29/07/2021
// Modified: 	15/11/2021
//==============================================================================

**************************
*** Importing the Data ***
**************************

clear

cd "XXXX"	// set the content directory here

* For Australian resident comparison
use HILDA_restricted_combined_rescaled_domestic_children

gen pw_net_worth = pwassei - pwdebti	// generating a personal net worth variable as the difference between personal assets and personal debts

gen married = 0
replace married = 1 if mrcurr == 1

gen death_age = hgage
replace death_age = 100 if death_age > 100

keep xwaveid wave hhpxid hgage hgsex hhwt* benchmark_ratio benchmarked* pw_net_worth death_age married	// keeping all relevant variables

destring xwaveid, replace
destring hhpxid, replace

**********************************
*** Merging in Death Rate Data ***
**********************************

gen year = wave + 2000	// generating a year variable to enable merging with the age and gender specific death rates

merge m:1 year death_age hgsex married using "Death_rates_by_marital_status.dta", nogen	// merging in the age and gender specific death rate data

*************************
*** Cleaning the Data ***
*************************

drop if (wave != 2 & wave != 6 & wave != 10 & wave != 14 & wave != 18)	//drops waves that don't have a wealth module

replace pw_net_worth = 0 if pw_net_worth < 0	// replacing negative values of net worth with zero because negative wealth cannot be passed on

* For Australian resident comparison
save HILDA_wealth_domestic_children, replace
save HILDA_partner_wealth_domestic_children, replace

**************************************
*** Weighting the Wealth Variables ***
**************************************

gen dw_net_worth = pw_net_worth * benchmarked_new_hhwte * deathrate

gen benchmarked_new_hhwte_deathrate = deathrate * benchmarked_new_hhwte

collapse(sum) dw_net_worth deathrate benchmarked_new_hhwte_deathrate benchmarked_new_hhwte, by(year)	// collapsing (summing) the weighted and unweighted wealth variables and their death-weighted counterparts as well as the weighted and unweighted death rates by year

* For Australian resident comparison
save HILDA_wealth_at_death_domestic_children, replace

keep year dw_net_worth deathrate benchmarked_new_hhwte_deathrate	// keeping variables that will be used for plotting

* For Australian resident comparison
save HILDA_estates_domestic_children, replace

*****************************************
*** Importing the Partner Wealth Data ***
*****************************************

clear

* For Australian resident comparison
use HILDA_partner_wealth_domestic_children

keep year xwaveid hhpxid deathrate	// keeping relevant identifiers and the death rate variable

drop if hhpxid == .	// dropping observations for those that do not have partners
drop hhpxid	// dropping the partner identifier variable

rename xwaveid hhpxid	// renaming the individual identifier to be the partner identifier
rename deathrate partner_deathrate	// renaming the death rate variable to allow it to be merged with the main dataset

* For Australian resident comparison
save HILDA_partner_deathrates_domestic_children, replace

************************************************
*** Merging the Partner Death Rate Variables ***
************************************************

clear

* For Australian resident comparison
use HILDA_wealth_domestic_children

* For Australian resident comparison
merge m:1 hhpxid year using HILDA_partner_deathrates_domestic_children

replace deathrate = deathrate * partner_deathrate if partner_deathrate != .	// for couples, replacing the likelihood of death as the likelihood that both members of a couple die in a given year
rename deathrate final_deathrate

**************************************
*** Weighting the Wealth Variables ***
**************************************

gen dw_final_net_worth = pw_net_worth * benchmarked_new_hhwte * final_deathrate

gen bm_final_new_hhwte_deathrate = final_deathrate * benchmarked_new_hhwte

collapse(sum) `wealth_vars' dw_final_net_worth final_deathrate bm_final_new_hhwte_deathrate benchmarked_new_hhwte, by(year)	// collapsing (summing) the weighted and unweighted final estate wealth variables and their death-weighted counterparts as well as the weighted and unweighted death rates by year

* For Australian resident comparison
save HILDA_WAD_final_domestic_children, replace

keep year dw_final_net_worth final_deathrate bm_final_new_hhwte_deathrate	// keeping variables that will be used for plotting

* For Australian resident comparison
save HILDA_final_estates_domestic_children, replace

********************************************
*** Merging with the All Estates Dataset ***
********************************************

* For Australian resident comparison
merge 1:1 year using HILDA_estates_domestic_children, nogen

************************************
*** Scaling Single Person Deaths ***
************************************

gen single_death_percent = bm_final_new_hhwte_deathrate / benchmarked_new_hhwte_deathrate	// generating a variable for single deaths as a proportion of total deaths

merge 1:1 year using Single_person_deaths, nogen	// merging in ABS data on single person deaths as a proportion of all deaths

gen scale_factor = abs_single_death_percent / single_death_percent	// calculating a scale factor to account for the under representation of single deaths in the HILDA dataset

local final_estate_vars dw_final_net_worth final_deathrate bm_final_new_hhwte_deathrate	// creating a local macro with all aggregate final estate variables and the single person death rates

foreach w in `final_estate_vars' {
	replace `w' = `w' * scale_factor
}	// scaling the final estate variables by the single person death percentage scaling factor

* For Australian resident comparison
save HILDA_all_estates_domestic_children, replace